/*! \file 
**********************************************************************************
*Title:                         Discretix OMA DRM v2 Toolkit ATP Test source file
*
* Filename:                    ODRM_TLK_ATP_aes_unwrap_and_store.c 
*
* 
* Created:                      09.12.2007
*
*
* \Author                      Sagit Ben Tsur
*
* \Remarks
*           Copyright (C) 2007 by Discretix Technologies Ltd. All Rights reserved.
**********************************************************************************/

/************* Include Files ***********************************/
#include "DX_VOS_BaseTypes.h"
#include "CRYS.h"
#include "tlk_odrm_types.h"
#include "KMNG_Defs.h" 
#include "ODRM_TLK_ATP_UTIL.h"
#include "KMNG_API.h"
#include "ODRM_TLK_ATP_data.h"
#include "tlk_odrm_api.h"
#include "DX_VOS_Mem.h"
#include "CRYS_KMNG.h"
#include "MW_ATP_UTIL_funcs.h"

/*****************************************************************************
* Function Name:                                                           
*  TST_ATP_TLK_ODRM_AesUnwrapAndStore
* 
* Inputs:
*  None
*
* Outputs:
*  DxError_t - Function error return
*
* Description:
*    Test TLK_ODRM_AesUnwrapAndStore
*
* Algorithm: 
*1. Get work space size. 
*2. Init the key Ring. 
*3. Insert wrapper key Kek to the Key Ring. 
*4. Load  parameters and call TLK_ODRM_AesUnwrapAndStore  unwraps the K and insert it to the Key Ring,
*   returns key id.
*5.	Activate  K from the key using  the key id received from the previous operation. 
*6. Validate this is the same key we inserted in step 4.
* 6.1  Encrypt Plain text with the K key from Key Ring ( from the id we received from the toolkit).
* 6.2 compare it to the Encrypted plain text with the original K.
*******************************************************************************/  
DxError_t TST_ATP_TLK_ODRM_AesUnwrapAndStore(void)
{

  DxError_t                   TST_TestStatus;
  DxError_t                   TST_Error;
  TLK_ODRM_KMNGKey_t          TLKKkey;
  TLK_ODRM_KMNGKey_t          TLKKekkey;
  KMNG_AES_WrappedKey_t       k_WrappedKey;
  KMNG_UserSpecificKeyData_t  UserSpecificKeyData = {0,0};
  TLK_ODRM_KMNGKeyRing_t      keyesRing;
  TLK_ODRM_Buffer_t           TLKwrappedKKeyBuff;

  TST_gNumOfSymKeys = 2;
  TST_gNumOfRSAKeys = 0;
  TST_gNumOfDHKeys  = 0;


  ATP_LOG_TST_PRINT((MW_ATP_MSG," ****************************************************** \n"));
  ATP_LOG_TST_PRINT((MW_ATP_MSG," ** TST_ATP_TLK_ODRM_AesUnwrapAndStore Test             \n"));     
  ATP_LOG_TST_PRINT((MW_ATP_MSG," ****************************************************** \n \n"));

  #ifdef  ODRM_TLK_CLEAR_KEY_RING
    DX_VOS_MemSetZero(TST_gKeyRingBuf,ODRMTLK_TST_MAX_SIZE_KEY_RING_BUF);
    DX_VOS_MemSetZero(TST_gDevKeyRingBuf,ODRMTLK_TST_MAX_SIZE_DEV_KEY_RING_BUF);
  #endif

  /*1. Get work space size */
  TST_Error = TLK_ODRM_WorkspaceSizeGet(TLK_ODRM_MIN_WORKSPACE,&TST_gWorkspaceSizeInBytes);

  TST_ATP_CHECK_ERROR_RETURN(TST_Error,"TLK_ODRM_WorkspaceSizeGet",
                                                "TST_ATP_TLK_ODRM_AesUnwrapAndStore",
                                                "TST_ATP_TLK_ODRM_AesUnwrapAndStore");

  
   if(TST_gWorkspaceSizeInBytes < ODRMTLK_TST_WORKSPACE_SIZE)
    TST_Error = TST_FAIL;
   else
    TST_Error = TST_PASS;
  
   
  TST_ATP_CHECK_ERROR_RETURN(TST_Error," Workspace Size ",
                                                "TST_ATP_TLK_ODRM_AesUnwrapAndStore","TST_ATP_TLK_ODRM_AesUnwrapAndStore");
 


  /************************************************************************/
  /*2. Init the key ring                                                   */
  /************************************************************************/
  
  /* Get key ring expected buffer size */
  TST_Error = KMNG_GetKeyRingBufferSize(TST_gNumOfSymKeys,  
                                        TST_gNumOfRSAKeys, 
                                        TST_gNumOfDHKeys,  
                                        &TST_gRequiredBufSizeInBytes); 

  TST_ATP_CHECK_ERROR_RETURN(TST_Error,"KMNG_GetKeyRingBufferSize",
                                                "TST_ATP_TLK_ODRM_AesUnwrapAndStore",
                                                "TST_ATP_TLK_ODRM_AesUnwrapAndStore");


  TST_Error = KMNG_KeyRingInit(TST_gPasswordSize,
                               TST_gPassword,
                               TST_gNumOfSymKeys,  
                               TST_gNumOfRSAKeys, 
                               TST_gNumOfDHKeys,  
                               TST_gKeyRingBuf);

  TST_ATP_CHECK_ERROR_RETURN(TST_Error,"KMNG_KeyRingInit",
                                                "TST_ATP_TLK_ODRM_AesUnwrapAndStore",
                                                "TST_ATP_TLK_ODRM_AesUnwrapAndStore");

 

  /************************************************************************/
  /* 3. Insert wrapper Kek key to the Key Ring                            */
  /************************************************************************/
  TST_Error = KMNG_ImportSymUserKey(TST_gPassword,
                                    TST_gPasswordSize,
                                    TST_gPassword,
                                    TST_gPasswordSize,
                                    KMNG_KeyTypeAES,
                                    ODRMTLK_TST_AES_KEY_SIZE,
                                    KMNG_KEY_USAGE_ODRM,
                                    TLK_ODRM_KEY_RESTRICTION,
                                    UserSpecificKeyData, 
                                    TST_gKekkey,
                                    &TLKKekkey.keyId,
                                    TST_gKeyRingBuf);


  TST_ATP_CHECK_ERROR_RETURN(TST_Error,"KMNG_ImportSymUserKey",
                                                "TST_ATP_TLK_ODRM_AesUnwrapAndStore",
                                                "TST_ATP_TLK_ODRM_AesUnwrapAndStore");

 
  /*******************************************************************************************************/
  /*4.Load  parameters and call TLK_ODRM_AesUnwrapAndStore  unwraps the K and insert it to the Key Ring,*/
  /* returns key id.                                                                                     */
  /*******************************************************************************************************/
  /************************************************************************/
  /* load key ring parameters                                             */
  /************************************************************************/
  keyesRing.keyKmngRing_ptr               = TST_gKeyRingBuf;
  keyesRing.keyKmngRingPassword_ptr       = TST_gPassword;
  keyesRing.keyKmngRingPasswordLenInBytes = TST_gPasswordSize;

  /************************************************************************/
  /* load Kek                                                             */
  /************************************************************************/
  TLKKekkey.keyPassword_ptr                = TST_gPassword;
  TLKKekkey.keyPasswordLenInBytes          = TST_gPasswordSize;

  /************************************************************************/
  /* load wrapped K  buff                                                 */
  /************************************************************************/
  TLKwrappedKKeyBuff.buff_ptr              = TST_gWrappedKkey;
  TLKwrappedKKeyBuff.buffSizeInBytes       = ODRMTLK_TST_WRAPPED_KEY_LENGTH_IN_BYTES;

  /************************************************************************/
  /* load  K  pass parameters                                             */
  /************************************************************************/
  TLKKkey.keyPassword_ptr               = TST_gPassword;
  TLKKkey.keyPasswordLenInBytes         = TST_gPasswordSize;

  TST_TestStatus = TLK_ODRM_AesUnwrapAndStore(&keyesRing,
                                              &TLKKekkey,
                                              &TLKwrappedKKeyBuff,
                                              &TLKKkey,
                                              TST_gWorkspace,
                                              ODRMTLK_TST_WORKSPACE_SIZE);

  TST_ATP_CHECK_ERROR_RETURN(TST_Error,"TLK_ODRM_AesUnwrapAndStore",
                                               "TST_ATP_TLK_ODRM_AesUnwrapAndStore",
                                               "TST_ATP_TLK_ODRM_AesUnwrapAndStore");

  
  /************************************************************************************** */
  /* 5. Activate  K from the key using  the key id received from the previous operation   */
  /************************************************************************************** */	

  /*prepares the K key for CRYS function */
  TST_Error = KMNG_ActivateAESKey(TST_gPassword,
                                  TST_gPasswordSize,
                                  TLKKkey.keyPassword_ptr, 
                                  TLKKkey.keyPasswordLenInBytes,
                                  TLKKkey.keyId,
                                  TST_gKeyRingBuf,
                                  k_WrappedKey);

  TST_ATP_CHECK_ERROR_RETURN(TST_Error,"KMNG_ActivateAESKey K",
                                               "TST_ATP_TLK_ODRM_AesUnwrapAndStore",
                                               "TST_ATP_TLK_ODRM_AesUnwrapAndStore");

  /************************************************************/
  /* 6. Validate this is the same key we inserted in step 4   */
  /************************************************************/	
  /*6.1  Encrypt Plain text with the  key from Key Ring ( from the id we received from the toolkit)*/
  TST_Error = CRYS_KMNG_AES(k_WrappedKey,
                            TST_gIvCounter,
                            CRYS_AES_Encrypt,
                            CRYS_AES_ECB_mode,
                            TST_gPlainBuff, 
                            ODRMTLK_TST_AES_KEY_SIZE,
                            TST_gEncryptedDataOut); 

   TST_ATP_CHECK_ERROR_RETURN(TST_Error,"CRYS_KMNG_AES  - k Key",
                                                 "TST_ATP_TLK_ODRM_AesUnwrapAndStore",
                                                 "TST_ATP_TLK_ODRM_AesUnwrapAndStore");

  
  /*6.2 compare it to the Encrypted plain text with the original K*/
  if(0 != DX_VOS_MemCmp(TST_gEncryptedPlainBuffWithK,TST_gEncryptedDataOut,ODRMTLK_TST_AES_KEY_SIZE))
       TST_Error = TST_FAIL; 
  else
    TST_Error = TST_PASS; 

  TST_ATP_CHECK_ERROR_RETURN(TST_Error,"DX_VOS_MemCmp - Kkey",
                                                "TST_ATP_TLK_ODRM_AesUnwrapAndStore",
                                                "TST_ATP_TLK_ODRM_AesUnwrapAndStore");

  

  ATP_LOG_TST_PRINT((MW_ATP_MSG," ***************************************************** \n\n"));
  ATP_LOG_TST_PRINT((MW_ATP_MSG," ** TST_ATP_TLK_ODRM_AesUnwrapAndStore  Test  PASS !!! \n"));     
  ATP_LOG_TST_PRINT((MW_ATP_MSG," ***************************************************** \n \n"));

EXIT_ON_ERROR:
  return TST_Error;
} 
